public class UnionFind
{
  private int [] iS;

  // Constructor
  public UnionFind(int numElements)
  {
    iS = new int[numElements];
    for (int i = 0; i < iS.length; i++)
      iS[i] = -1;  // root of one-element tree
  }
  
  // Take union of two disjoint sets using the size heuristic.
  // root1 and root2 must be distinct, and must represent sets.
  public void union(int root1, int root2)
  {
    assertIsRoot(root1);
    assertIsRoot(root2);
    if (root1 == root2)
      throw new IllegalArgumentException("Union: root1 == root2 " + root1);
    
    if (iS[root2] < iS[root1]) { 
      // root2 is larger --> root2 becomes new root
      iS[root2] += iS[root1];
      iS[root1] = root2; 
    } else {
      // root1 is larger --> root1 becomes new root
      iS[root1] += iS[root2];
      iS[root2] = root1;
    }
  }
  
  // Perform a find with path compression.
  public int find(int x)
  {
    assertIsItem(x);
    if (iS[x] < 0)
      return x;
    else
      return iS[x] = find(iS[x]);
  }
  
  private void assertIsRoot(int root)
  {
    assertIsItem(root);
    if (iS[root] >= 0)
      throw new IllegalArgumentException("Union: " + root + " not a root");
  }
    
  private void assertIsItem(int x)
  {
    if (x < 0 || x >= iS.length)
      throw new IllegalArgumentException("UnionFind: " + x + " not an item");
  }

  // --------------------------------------------------------------------
  
  // Test main; all finds on same output line should be identical
  public static void main(String [] args)
  {
    int numElements = 128;
    int numInSameSet = 16;
    
    UnionFind ds = new UnionFind(numElements);
    int set1, set2;
    
    for (int k = 1; k < numInSameSet; k *= 2) {
      for (int j = 0; j + k < numElements; j += 2 * k) {
	set1 = ds.find(j);
	set2 = ds.find(j + k);
	ds.union(set1, set2);
      }
    }
    
    for (int i = 0; i < numElements; i++) {
      System.out.print(ds.find(i)+ "*");
      if (i % numInSameSet == numInSameSet - 1)
	System.out.println();
    }
    System.out.println();
  }
}
