import java.awt.*;
import java.awt.event.WindowEvent;
import java.awt.event.KeyEvent;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseEvent;
import java.awt.Point;

public class Maze extends Frame
{
  boolean iAll;
  int iRows, iCols;
  boolean [][][] iMaze;

  public Maze(int rows, int cols)
  {
    iRows = rows;
    iCols = cols;
    iAll = true;
    toggleAll();
    iMaze = new boolean[2][iRows][iCols];
    setSize(800, 350);
    addWindowListener(new AppCloser());
    addComponentListener(new Resizer(this));
    addKeyListener(new KeyHandler(this));
    buildMaze();
  }
  
  public void paint(Graphics g)
  {
    if (!(g instanceof Graphics2D))
      return;
    Graphics2D gg = (Graphics2D) g;

    Dimension d = getSize();
    gg.setColor(Color.white);
    gg.fillRect(0,0, d.width, d.height);

    Insets insets = getInsets();
    int x0 = insets.left + 1;
    int y0 = insets.top + 1;
    int rowWid = (d.height - y0 - insets.bottom - 2) / iRows;
    int colWid = (d.width - x0 - insets.right - 2) / iCols;
    int wid = rowWid < colWid ? rowWid : colWid;

    gg.setColor(Color.black);
    gg.drawRect(x0, y0, iCols * wid, iRows * wid);

    for (int row = 0; row < iRows; ++row) {
      for (int col = 0; col < iCols; ++col) {
	int x = x0 + col * wid;
	int y = y0 + row * wid;
	if (col < iCols - 1 && iMaze[0][row][col])
	  gg.drawLine(x + wid, y, x + wid, y + wid);
	if (row < iRows -1 && iMaze[1][row][col])
	  gg.drawLine(x, y + wid, x + wid, y + wid);
      }
    }
  }

  private static <AnyType> void randomPermutation(AnyType [] A)
  {
    java.util.Random random = new java.util.Random();
    for (int i = A.length; i > 1; --i) {
      int j = random.nextInt(i);
      AnyType t = A[i-1];
      A[i-1] = A[j];
      A[j] = t;
    }
  }

  private static class Wall {
    public Wall(int dir, int row, int col) { 
      iDir = dir; iRow = row; iCol = col; }
    int iDir, iRow, iCol;
  }

  public void buildMaze()
  {
    // make all walls
    for (int dir = 0; dir < 2; ++dir) {
      for (int row = 0; row < iRows; ++row) {
	for (int col = 0; col < iCols; ++col) {
	  iMaze[dir][row][col] = true;
	}
      }
    }

    // make array of all walls
    Wall [] wall = new Wall[iRows * (iCols - 1) + (iRows - 1) * iCols];
    int i = 0;
    for (int row = 0; row < iRows; ++row) {
      for (int col = 0; col < iCols - 1; ++col)
	wall[i++] = new Wall(0, row, col);
    }
    for (int row = 0; row < iRows - 1; ++row) {
      for (int col = 0; col < iCols; ++col)
	wall[i++] = new Wall(1, row, col);
    }
    
    randomPermutation(wall);
    
    // make union-find structure
    UnionFind sets = new UnionFind(iRows * iCols);

    int src = 0;
    int dst = iRows * iCols - 1;
    for (i = 0; i < wall.length; ++i) {
      // cells separated by this wall
      Wall w = wall[i];
      int n1 = w.iRow * iCols + w.iCol;
      int n2 = (w.iDir > 0) ? (n1 + iCols) : (n1 + 1);
      int set1 = sets.find(n1);
      int set2 = sets.find(n2);
      if (set1 != set2) {
	// remove wall
	iMaze[w.iDir][w.iRow][w.iCol] = false;
	sets.union(set1, set2);
      }
      if (!iAll && (sets.find(src) == sets.find(dst)))
	break;
    }
  }
  
  public void toggleAll()
  {
    iAll = !iAll;
    String s = iAll? "(all cells connected)" : 
      "(source-destination connected)";
    setTitle("Maze " + iRows + " x " + iCols + " " + s);
  }

  // --------------------------------------------------------------------

  static class AppCloser extends java.awt.event.WindowAdapter {
    public void windowClosing(WindowEvent e) {
      System.exit(0);
    }
  }

  static class Resizer extends java.awt.event.ComponentAdapter {
    public Resizer(Maze g) { iParent = g; }

    public void componentResized(ComponentEvent e)
    {
      iParent.repaint();
    }
    private Maze iParent;
  }
  
  static class KeyHandler extends java.awt.event.KeyAdapter {
    public KeyHandler(Maze v) { iParent = v; }

    public void keyTyped(KeyEvent e) {
      char ch = e.getKeyChar();
      switch (ch) {
      case 'a':
	iParent.toggleAll();
      case 'n':
	iParent.buildMaze();
	iParent.repaint();
	break;
      case 'q':
	System.exit(0);
      }
    }
    private Maze iParent;
  }

  // --------------------------------------------------------------------

  public static void main(String [] args)
  {
    if (args.length != 0 && args.length != 2) {
      System.err.println("Usage: java Maze [ <rows> <columns> ]");
      System.exit(1);
    }
    int rows = 3;
    int cols = 8;
    if (args.length == 2) {
      rows = Integer.parseInt(args[0]);
      cols = Integer.parseInt(args[1]);
    }

    Maze maze = new Maze(rows, cols);
    maze.setVisible(true);
  }

  // --------------------------------------------------------------------
}
